/**  @file bta_filters.c
*  
*    @brief The wrapper for all the filter implementations
*  
*    BLT_DISCLAIMER
*  
*    @author Alex Falkensteiner
*  
*    @cond svn
*  
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*  
*    @endcond
*/

#include <bta.h>
#include <bta_helper.h>
#include <bta_oshelper.h>
#include "bta_filters_private.h"
#include "bta_avg_sequences.h"
#include "bta_avg_pixels.h"
#include "bta_combine.h"
#include "bta_crop.h"
#include "bta_crop_chessboard.h"
#include "bta_find_chessboard.h"
#include "bta_dct.h"
#include "bta_focus_preproc.h"
#include "bta_gaussian_blur.h"
#include "bta_intrinsic_undistort.h"
#include "bta_laplace.h"
#include "bta_math.h"
#include "bta_morphology.h"
#include "bta_motion_detector.h"
#include "bta_orientation.h"
#include "bta_pixel_intrpl.h"
#include "bta_rms_frames.h"
#include <stdlib.h>

static void *lock = 0;



BTA_Status BFLTaddFilter(BTA_FltHandle **filterHandles, uint16_t *filterHandlesLen, BTA_FltType **fltTypes, BTA_FltConfig *fltConfig, BTA_FltType fltType, BTA_InfoEventInst *infoEventInst) {
    if (!filterHandles || !filterHandlesLen || !fltTypes) {
        return BTA_StatusInvalidParameter;
    }
    if (!lock) {
        BTAinitMutex(&lock);
    }
    BTAlockMutex(lock);
    if (!*filterHandles) {
        *filterHandles = (BTA_FltHandle *)malloc(sizeof(BTA_FltHandle));
        if (!*filterHandles) {
            BTAunlockMutex(lock);
            return BTA_StatusOutOfMemory;
        }
        *fltTypes = (BTA_FltType *)malloc(sizeof(BTA_FltType));
        if (!*fltTypes) {
            free(*filterHandles);
            *filterHandles = 0;
            BTAunlockMutex(lock);
            return BTA_StatusOutOfMemory;
        }
    }
    else {
        BTA_FltHandle *temp1 = *filterHandles;
        BTA_FltType *temp2 = *fltTypes;
        *filterHandles = (BTA_FltHandle *)realloc(*filterHandles, (*filterHandlesLen + 1) * sizeof(BTA_FltHandle));
        if (!*filterHandles) {
            *filterHandles = temp1;
            BTAunlockMutex(lock);
            return BTA_StatusOutOfMemory;
        }
        *fltTypes = (BTA_FltType *)realloc(*fltTypes, (*filterHandlesLen + 1) * sizeof(BTA_FltType));
        if (!*fltTypes) {
            *fltTypes = temp2;
            BTAunlockMutex(lock);
            return BTA_StatusOutOfMemory;
        }
    }
    BTA_FltHandle fltHandle;
    BTA_Status status;
    switch (fltType) {
    case BTA_FltTypeIntrinsicUndistort:
        status = BFLTintrinsicUndistortInit((BTA_FltIntrinsicUndistortConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeAvgSequences:
        status = BFLTavgSequencesInit((BTA_FltAvgSequencesConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeMath:
        status = BFLTmathInit((BTA_FltMathConfig *)fltConfig, &fltHandle, infoEventInst);
        break;

#ifndef BTA_EXCLUDE_FILTERS
    case BTA_FltTypeAvgPixels:
        status = BFLTavgPixelsInit((BTA_FltAvgPixelsConfig*)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeMotionDetector:
        status = BFLTmotionDetectorInit((BTA_FltMotionDetectorConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypePixelIntrpl:
        status = BFLTpixelIntrplInit((BTA_FltPixelIntrplConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeMorphology:
        status = BFLTmorphologyInit((BTA_FltMorphologyConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeOrientation:
        status = BFLTorientationInit((BTA_FltOrientationConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeDct:
        status = BFLTdctInit((BTA_FltDctConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeCropChessboard:
        status = BFLTcropChessboardInit((BTA_FltCropChessboardConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeFindChessboard:
        status = BFLTfindChessboardInit((BTA_FltFindChessboardConfig*)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeLaplace:
        status = BFLTlaplaceInit((BTA_FltLaplaceConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeFocusPreproc:
        status = BFLTfocusPreprocInit((BTA_FltFocusPreprocConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeCrop:
        status = BFLTcropInit((BTA_FltCropConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeCombine:
        status = BFLTcombineInit((BTA_FltCombineConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeGaussianBlur:
        status = BFLTgaussianBlurInit((BTA_FltGaussianBlurConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
    case BTA_FltTypeRmsFrames:
        status = BFLTrmsFramesInit((BTA_FltRmsFramesConfig *)fltConfig, &fltHandle, infoEventInst);
        break;
#endif
    default:
        BTAinfoEventHelper(infoEventInst, 1, BTA_StatusNotSupported, "BFLTinit: unsupported filter", 0);
        status = BTA_StatusNotSupported;
        break;
    }
    if (status != BTA_StatusOk) {
        BTAunlockMutex(lock);
        return status;
    }
    (*filterHandles)[*filterHandlesLen] = fltHandle;
    (*fltTypes)[*filterHandlesLen] = fltType;
    (*filterHandlesLen)++;
    BTAunlockMutex(lock);
    return BTA_StatusOk;
}


BTA_Status BFLTapplyFilters(BTA_FltHandle *fltHandles, uint16_t fltHandlesLen, BTA_FltType *fltTypes, BTA_Frame **frame, BTA_InfoEventInst *infoEventInst) {
    int i;
    BTA_Status status;
    if (!frame) {
        return BTA_StatusInvalidParameter;
    }
    if (!fltHandles || !fltTypes || fltHandlesLen <= 0) {
        // no filter to apply
        return BTA_StatusOk;
    }
    if (!lock) {
        return BTA_StatusRuntimeError;
    }
    BTAlockMutex(lock);
    for (i = 0; i < fltHandlesLen; i++) {
        switch (fltTypes[i]) {
        case BTA_FltTypeIntrinsicUndistort:
            status = BFLTintrinsicUndistortApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeAvgSequences:
            status = BFLTavgSequencesApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeMath:
            status = BFLTmathApply(fltHandles[i], frame);
            break;
#ifndef BTA_EXCLUDE_FILTERS
        case BTA_FltTypeAvgPixels:
            status = BFLTavgPixelsApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeMotionDetector:
            status = BFLTmotionDetectorApply(fltHandles[i], frame);
            break;
        case BTA_FltTypePixelIntrpl:
            status = BFLTpixelIntrplApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeMorphology:
            status = BFLTmorphologyApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeOrientation:
            status = BFLTorientationApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeDct:
            status = BFLTdctApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeCropChessboard:
            status = BFLTcropChessboardApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeFindChessboard:
            status = BFLTfindChessboardApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeLaplace:
            status = BFLTlaplaceApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeFocusPreproc:
            status = BFLTfocusPreprocApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeCrop:
            status = BFLTcropApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeCombine:
            status = BFLTcombineApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeGaussianBlur:
            status = BFLTgaussianBlurApply(fltHandles[i], frame);
            break;
        case BTA_FltTypeRmsFrames:
            status = BFLTrmsFramesApply(fltHandles[i], frame);
            break;
#endif
        default:
            status = BTA_StatusNotSupported;
            break;
        }
        BTAunlockMutex(lock);


        if (status != BTA_StatusOk) {
            return status;
        }
        if (!frame) {
            // The filter swallowed the frame. Img proc chain interrupted. Filter apparently needs more input frames. Done here!
            return BTA_StatusOk;
        }
    }
    return BTA_StatusOk;
}


BTA_Status BFLTremoveFilter(BTA_FltHandle **fltHandles, uint16_t *fltHandlesLen, BTA_FltType **fltTypes, BTA_FltHandle fltHandle, BTA_InfoEventInst *infoEventInst) {
    if (!fltHandles || !fltHandlesLen || !fltTypes) {
        return BTA_StatusInvalidParameter;
    }
    if (!*fltHandles || !*fltHandlesLen || !*fltTypes) {
        return BTA_StatusIllegalOperation;
    }
    if (!lock) {
        return BTA_StatusRuntimeError;
    }
    BTAlockMutex(lock);
    if (!fltHandle) {
        // No specific handle specified -> remove last filter
        (*fltHandlesLen)--;
        switch ((*fltTypes)[*fltHandlesLen]) {
        case BTA_FltTypeIntrinsicUndistort:
            BFLTintrinsicUndistortClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeAvgSequences:
            BFLTavgSequencesClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeMath:
            BFLTmathClose(&((*fltHandles)[*fltHandlesLen]));
            break;
#ifndef BTA_EXCLUDE_FILTERS
        case BTA_FltTypeAvgPixels:
            BFLTavgPixelsClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeMotionDetector:
            BFLTmotionDetectorClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypePixelIntrpl:
            BFLTpixelIntrplClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeMorphology:
            BFLTmorphologyClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeOrientation:
            BFLTorientationClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeDct:
            BFLTdctClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeCropChessboard:
            BFLTcropChessboardClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeFindChessboard:
            BFLTfindChessboardClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeLaplace:
            BFLTlaplaceClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeFocusPreproc:
            BFLTfocusPreprocClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeCrop:
            BFLTcropClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeCombine:
            BFLTcombineClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeGaussianBlur:
            BFLTgaussianBlurClose(&((*fltHandles)[*fltHandlesLen]));
            break;
        case BTA_FltTypeRmsFrames:
            BFLTrmsFramesClose(&((*fltHandles)[*fltHandlesLen]));
            break;
#endif
        default:
            BTAinfoEventHelper(infoEventInst, 1, BTA_StatusNotSupported, "BFLTclose: unsupported filter", 0);
            break;
        }
        (*fltHandles)[*fltHandlesLen] = 0;
        (*fltTypes)[*fltHandlesLen] = (BTA_FltType)0;
        if (!*fltHandlesLen) {
            free(*fltHandles);
            *fltHandles = 0;
            free(*fltTypes);
            *fltTypes = 0;
        }
        BTAunlockMutex(lock);
        if (!*fltHandlesLen) {
            BTAcloseMutex(lock);
            lock = 0;
        }
        return BTA_StatusOk;
    }
    // TODO: implement removal by handle
    BTAunlockMutex(lock);
    return BTA_StatusNotSupported;
}